;;; forwarder.lisp --- Server part of the Rfb-over-HTTP tunnel.
;;;
;;; $Id: forwarder.lisp,v 1.8 2003/12/11 21:39:20 helmut Exp $
;;;
;;;   Copyright (C) 2003  Helmut Eller <helmut@online-marketwatch.com>
;;;
;;; Part of HttpTunnel4vnc
;;; (see http://www.online-marketwatch.com/HttpTunnel4vnc/)
;;;
;;; The forwarder listens on a port and waits for HTTP requests from
;;; the Java applets and copies the body of the request to the VNC
;;; server.  
;;;
;;; The applet first request a session-id with the GET /get-session-id
;;; request.  The forwarder then opens a connection to the VNC server
;;; and returns a session-id to the applet.  The connection associated
;;; with the session-id is stored in the global table *vnc-sockets*.
;;;
;;; All following requests by the applet encode the session-id in the
;;; url, e.g. GET /vnc-session?id=10 HTTP/1.1.  The forwarder uses the
;;; session-id to find the corresponding socket, and forwards the
;;; request body to the VNC server.
;;; 
;;; There is one thread per connection.  Connections are persistent
;;; (keep-alive) per default, and can be used for multiple request.
;;; If the applet uses keep-alive connections, e.g. appletviewer does
;;; that, only two threads are created: one connection for POST one
;;; connection for GET requests.
;;;

;;; Todo:
;;; 
;;; - Provide a way to kill sessions.
;;; - Figure out why there is still some consing.

(defpackage :forwarder
  (:use :common-lisp :ext :unix :c-call :alien)
  (:import-from :mp :make-process))

(in-package :forwarder)

(defmacro ldebug (level fstring &rest args)
  (if nil ;(member level '(infrequent accept get post header connection))
      `(progn (format *error-output* ,fstring . ,args))
      '(progn)))

;;;; Defresource:

;;; A "resource" avoids consing and gc overhead by reusing objects
;;; with known lifetimes.  The code is stolen from PAIP.

(defmacro defresource (name &key constructor (initial-copies 0)
                       (size (max initial-copies 10)))
  (let ((resource (symbolicate '* name '-resource*))
        (deallocate (symbolicate 'deallocate- name))
        (allocate (symbolicate 'allocate- name)))
    `(progn
       (defparameter ,resource (make-array ,size :fill-pointer 0))
       (defun ,allocate ()
         "Get an element from the resource pool, or make one."
         (if (= (fill-pointer ,resource) 0)
             ,constructor
             (vector-pop ,resource)))
       (defun ,deallocate (,name)
         "Place a no-longer-needed element back in the pool."
         (vector-push-extend ,name ,resource))
       ,(if (> initial-copies 0)
            `(mapc #',deallocate (loop repeat ,initial-copies 
                                       collect (,allocate))))
       ',name)))

(defmacro with-resource ((var resource &optional protect) &rest body)
  "Execute body with VAR bound to an instance of RESOURCE."
  (let ((allocate (symbolicate 'allocate- resource))
        (deallocate (symbolicate 'deallocate- resource)))
    (if protect
        `(let ((,var (,allocate)))
	  (unwind-protect (progn . ,body)
	    (,deallocate ,var)))
        `(let ((,var (,allocate)))
	  (multiple-value-prog1 (progn ,@body)
	    (,deallocate ,var))))))

(defconstant +buffer-length+ 50000)

(defresource string-buffer 
    :constructor (make-string +buffer-length+)
    :size 100
    :initial-copies 5)

(defconstant +line-length+ 300)

(defresource line-buffer
    :constructor (make-string +line-length+) 
    :size 100 
    :initial-copies 5)

(defconstant +crlf+ (coerce '(#\return #\linefeed) 'string)
  "HTTP header-lines are terminated with \r\n.")

(defstruct header request lines)

(defun only-whitespace-p (line)
  "Return non-nil when the string LINE contains no characters other
than #\space #\tab #\newline #\return."
  (every (lambda (c) 
	   (find c '(#\space #\tab #\newline #\return)))
	 line))

(defun read-header-line (stream buffer)
  "Like read-line but store the the line into BUFFER.  Assume that the
BUFFER is large enough to hold an entire line."
  (declare (type stream stream)
	   (type simple-string buffer)
	   (optimize (speed 3)))
  (let ((max (length buffer)))
    (iterate rec ((i 0))
      (declare (type (integer 0 #.+line-length+) i))
      (if (= i max)
	  (error "Buffer to small.")
	  (let ((char (read-char stream)))
	    (cond ((char= char #\linefeed)
		   i)
		  (t
		   (setf (schar buffer i) char)
		   (rec (1+ i)))))))))

(defun parse-request-header (stream)
  "Parse the HTTP header and return four values:
COMMAND SESSION-ID HTTP-VERSION KEEP-ALIVE CONTENT-LENGTH

  COMMAND: one of the symbols  GET-SESSION-ID, GET, or POST.
  SESSION-ID: a fixnum or nil if COMMAND is get-session-id.
  HTTP-VERSION: either 0 or 1.
  KEEP-ALIVE: a boolean.
  CONTENT-LENGTH: nil or a fixnum if COMMAND is post."
  (with-resource (line line-buffer)
    (labels ((whitespace? (char)
	       (case char ((#\space #\tab #\newline #\return) t)))
	     (empty-line? (end)
	       (loop for i below end
		     always (whitespace? (schar line i))))
	     (next-token (start end)
	       (let* ((s (do ((s start (1+ s)))
			     ((or (= s end)
				  (not (whitespace? (schar line s))))
			      s)))
		      (e (do ((e s (1+ e)))
			     ((or (= e end)
				  (whitespace? (schar line e)))
			      e))))
		 (values s e)))
	     (read-request-line ()
	       (loop (let ((end (read-header-line stream line)))
		       (unless (empty-line? end)
			 (return end)))))
	     (match? (pattern line-end)
	       (string= pattern line :end2 (min (length pattern) line-end)))
	     (parse-id (start end)
	       (multiple-value-bind (id pos) 
		   (parse-integer line :start start :end end :junk-allowed t)
		 (unless id
		   (error "Invalid session-id: ~S" (subseq line start end)))
		 (values id pos)))
	     (parse-http-version (start line-end)
	       (multiple-value-bind (start end) (next-token start line-end)
		 (cond ((string= "HTTP/1.1" line :start2 start :end2 end) 1)
		       ((string= "HTTP/1.0" line :start2 start :end2 end) 0)
		       (t (error "Unsupported HTTP version: ~S" 
				 (subseq line start end))))))
	     (token= (string start end)
	       (string-equal string line :start2 start :end2 end))
	     (aux (command start line-end)
	       (multiple-value-bind (id end) (parse-id start line-end)
		 (let ((http-version (parse-http-version end line-end)))
		   (let ((keep-alive (= http-version 1))
			 (content-length nil))
		     (loop
		      (setq line-end (read-header-line stream line))
		      (when (empty-line? line-end)
			(return))
		      (multiple-value-bind (start end) (next-token 0 line-end)
			(cond ((token= "Connection:" start end)
			       (multiple-value-bind (start end)
				   (next-token end line-end)
				 (cond ((token= "close" start end)
					(setq keep-alive nil))
				       ((token= "keep-alive" start end)
					(setq keep-alive t))
				       (t 
					(error
					 "Unsupported connection-token: ~S"
					 (subseq line start end))))))
			      ((token= "Content-length:" start end)
			       (setq content-length
				     (parse-integer line :start end
						    :end line-end))))))
		     (values command id http-version keep-alive content-length)
		     )))))
      (let ((line-end (read-request-line)))
	(cond ((match? "GET /get-session-id " line-end)
	       (let ((http-version (parse-http-version 
				    (length "GET /get-session-id ")
				    line-end)))
		 (values 'get-session-id nil http-version nil nil)))
	      ((match? "GET /vnc-session?id=" line-end)
	       (aux 'get (length "GET /vnc-session?id=") line-end))
	      ((match? "POST /vnc-session?id=" line-end)
	       (aux 'post (length "POST /vnc-session?id=") line-end))
	      (t
	       (error "Unsupported HTTP-method: ~A" line)))))))
		     
(defun test-header-parser (header-strings &rest expected-result)
  (with-input-from-string (input
			   (with-output-to-string (stream)
			     (dolist (string header-strings)
			       (format stream "~A~A" string +crlf+))
			     (write-string +crlf+ stream)))
    (let ((result (multiple-value-list (parse-request-header input))))
      (cond ((equal result expected-result)
	     t)
	    (t (format t "~&expected: ~A~%" expected-result)
	       (format t "~&actual  : ~A~%" result)
	       nil)))))

#|
(test-header-parser '("GET /get-session-id HTTP/1.1")
		    'get-session-id nil 1 nil nil)
(test-header-parser '("GET /get-session-id HTTP/1.0")
		    'get-session-id nil 0 nil nil)
(test-header-parser '("GET /vnc-session?id=23 HTTP/1.0")
		    'get 23 0 nil nil)
(test-header-parser '("GET /vnc-session?id=23 HTTP/1.1")
		    'get 23 1 t nil)
(test-header-parser '("GET /vnc-session?id=23 HTTP/1.1"
		      "Connection: close")
		    'get 23 1 nil nil)
(test-header-parser '("GET /vnc-session?id=23 HTTP/1.1"
		      "Connection: keep-alive")
		    'get 23 1 t nil)
(test-header-parser '("POST /vnc-session?id=23 HTTP/1.1"
		      "Content-length: 234"
		      "Connection: close"
		      )
		    'post 23 1 nil 234)
|#

(declaim (inline read-n-bytes-non-blocking))
(defun read-n-bytes-non-blocking (buffer stream)
  "Try to fill BUFFER with bytes read from STREAM.  This is a
thread-aware variant of sys:read-n-bytes."
  (declare (optimize speed))
  (mp:process-wait-until-fd-usable (sys:fd-stream-fd stream) :input)
  (sys:read-n-bytes stream buffer 0 (length buffer) nil))

(defmacro write-header-lines (stream &rest strings)
  (let ((out '#:stream))
    `(let ((,out ,stream))
	(declare (type stream ,out) 
		 (optimize (speed 3)))
	(write-string  , (with-output-to-string (*standard-output*)
			   (loop for string in strings
				 do (format t "~A~A" string +crlf+)))
	 ,out))))

(defun get-request (client vnc-server http-version)
  "Serve a GET request, i.e. copy the available output available from
VNC-SERVER to CLIENT."
  (declare (type stream client vnc-server))
  (with-resource (buffer string-buffer)
    (let ((buffer buffer))
      (declare (type simple-string buffer)
	       (optimize (speed 3)))
      (labels ((http-1.1 ()
		 (write-header-lines client
				     "HTTP/1.1 200 OK"
				     "Cache-control: no-cache")
		 (let ((count (read-n-bytes-non-blocking buffer vnc-server)))
		   (declare (type (integer 0 #.+buffer-length+) count))
		   ;; Use chunked transfer-encoding if the output
		   ;; doesn't fit in the buffer.  This happens rarely
		   ;; because the buffer is 50000 bytes large.
		   (cond ((< count (length buffer))
			  (write-string "Content-length: " client)
			  ;; Avoids consing-overhead for (format
			  ;; client "~D" count)
			  (lisp::sub-output-integer count client)
			  (write-header-lines client "" "")
			  (write-string buffer client :end count)
			  (force-output client)
			  (ldebug get "~&[get][~D]~%" count)
			  (ldebug get-verbose "~&[get][~D]<~A>~A~%" 
				  count
				  (subseq buffer 0 count) 
				  client))
			 (t
			  (write-header-lines client 
					      "Transfer-Encoding: chunked" "")
			  (loop (format client "~X~A" count +crlf+)
				(write-sequence buffer client :end count)
				(write-header-lines client "")
				(ldebug get "~&[get][chunk ~D]~%" count)
				(when (< count (length buffer))
				  (return))
				(setf count (read-n-bytes-non-blocking 
					     buffer vnc-server)))
			  (write-header-lines client "0" "")
			  (force-output client)))))
	       (http-1.0 ()
		 (write-header-lines client 
				     "HTTP/1.0 200 OK"
				     "Pragma: no-cache")
		 ;; can't use chunked transfer encodeing.
		 (let ((count (read-n-bytes-non-blocking buffer vnc-server)))
		   (write-string "Content-length: " client)
		   (lisp::sub-output-integer count client)
		   (write-header-lines client "" "")
		   (write-sequence buffer client :end count)
		   (force-output client))))
	(ecase http-version
	  (1 (http-1.1))
	  (0 (http-1.0)))))))

(defun post-n-bytes (in n out)
  "Copy N bytes form IN to OUT."
  (declare (type stream in out)
	   (type fixnum n))
  (with-resource (buffer line-buffer)
    (iterate rec ((pending n))
      (cond ((zerop pending)
	     (ldebug post "~&[post][~D]~%" n)
	     (force-output out))
	    (t
	     (let ((size (min +line-length+ pending)))
	       (read-sequence buffer in :end size)
	       (write-sequence buffer out :end size)
	       (rec (- pending size))))))))

(defun post-request (client length vnc-socket http-version)
  "Handle a post-request with content-length LENGTH."
  (declare (optimize speed)
	   (type fixnum length)
	   (type stream client vnc-socket))
  (ecase http-version
    (1 (write-header-lines client "HTTP/1.1 200 OK"))    
    (0 (write-header-lines client "HTTP/1.1 200 OK")))
  (write-header-lines client "Content-length: 0" "")
  (force-output client)
  (post-n-bytes client length vnc-socket))
	     
(defparameter *vnc-sockets* '()
  "List of connections to the VNC server.  Each element is a cons of
the form (session-id . socket)." )

(defun find-session-socket (session-id)
  "Find the socket for SESSION-ID."
  (cdr (assoc session-id *vnc-sockets*)))

(defun lookup-session-socket (session-id)
  "Find the socket for SESSION-ID and signal an error if there is no
such session."
  (let ((socket (find-session-socket session-id)))
    (unless socket
      (error "No session for id: ~A" session-id))
    socket))

(defun register-session-socket (socket)
  "Register a session for SOCKET an return a session-id."
  (let ((id (sys:fd-stream-fd socket)))
    (assert (not (find-session-socket id)))
    (push (cons id socket) *vnc-sockets*)
    id))

(defun deregister-session-socket (socket)
  (let ((id (cdr (rassoc socket *vnc-sockets*))))
    (ldebug infrequent "~&Closing session: ~A~%" id)
    (close socket)
    (setf *vnc-sockets* (delete socket *vnc-sockets* :key #'cdr))))

(defun ip->dotted (ip)
  (format nil "~D.~D.~D.~D"
	  (ldb (byte 8 24) ip)
	  (ldb (byte 8 16) ip)
	  (ldb (byte 8 8)  ip)
	  (ldb (byte 8 0)  ip)))

;;; ffi binding for strftime.

(defmacro with-growing-buffer ((buffer size &key (initial-size 1024))
			       &body body)
  "Execute BODY repeatedly with BUFFER bound to an alien of type
 (array char *) and size SIZE.  SIZE is doubled on each iteration."
  `(flet ((try (,buffer ,size) ,@body))
    (declare (inline try)) 
    (let ((size ,initial-size))
      (with-alien ((buffer (array char ,initial-size)))
	(try (cast buffer (* char)) ,initial-size))
      (loop 
       (setf size (* 2 size))
       (let ((buffer (make-alien char size)))
	 (unwind-protect 
	      (try buffer size)
	   (free-alien buffer)))))))

(def-alien-type tm (struct unix::tm))
(def-alien-type time-t unix::time-t)

(defun format-time-string (format-string &key time universal)
  (with-alien ((utime time-t)
	       (brokentime tm)
	       (c/time (function time-t (* time-t)) 
		       :extern "time")
	       (c/gmtime_r (function (* tm) (* time-t) (* tm))
			   :extern "gmtime_r")
	       (c/localtime_r (function (* tm) (* time-t) (* tm))
			      :extern "localtime_r")
	       (c/strftime (function int (* char) int c-string (* tm))
			   :extern "strftime"))
    (setf utime (or (and time 
			 (- time 
			      #.(encode-universal-time 0 0 0 1 1 1970 0)))
		    (let ((time (alien-funcall c/time nil)))
		      (if (= time -1)
			  (error "c/time failed: ~A" (get-unix-error-msg))
			  time))))
    (let ((result (alien-funcall (if universal c/gmtime_r c/localtime_r)
				 (addr utime) 
				 (addr brokentime))))
      (unless (sys:sap= (alien-sap result)
			(alien-sap (addr brokentime)))
	(error "format-time-string failed")))
    (with-growing-buffer (buffer size :initial-size 256)
      (setf (deref buffer 0) 1)
      (let ((count (alien-funcall c/strftime 
				  buffer size format-string
				  (addr brokentime))))
	(when (or (plusp count)
		  (and (zerop count)
		       (= (deref buffer 0) 0)))
	  (let ((string (make-string count)))
	    (kernel:copy-from-system-area 
	     (alien-sap buffer) 
	     0
	     string 
	     (* vm:vector-data-offset vm:word-bits)
	     (* count vm:byte-bits))
	    (return-from format-time-string string)))))))

(defun set-file-mode (fd mode)
  "Set the file permissions for filedescriptor FD to MODE."
  (let ((old-mode (logand #o777 (nth-value 3 (unix-fstat fd)))))
    (unless (= mode old-mode)
      (multiple-value-bind (ok errno) (unix-fchmod fd mode)
	(unless ok
	  (error "unix-fchmod failed: ~A" (get-unix-error-msg errno)))))))

(defun log-session-start (client id)
  "Write some info for the new session to the log-file."
  (with-open-file (*standard-output* "/tmp/forwarder-log"
				     :direction :output
				     :if-exists :append
				     :if-does-not-exist :create)
    (set-file-mode (sys:fd-stream-fd *standard-output*) #o664)
    (let ((socket (sys:fd-stream-fd client)))
      (format t  "~2&Starting session: ~A~%" id)
      (format t   "~&    time: ~A~%" (format-time-string "%d.%m.%Y %H:%M:%S"))
      (multiple-value-bind (host port) (get-socket-host-and-port socket)
	(format t "~&   local: ~D:~D~%" (ip->dotted host) port))
      (multiple-value-bind (host port) (get-peer-host-and-port socket)
	(format t "~&  remote: ~D:~D~%" (ip->dotted host) port))
      (loop for line = (read-line client)
	    until (only-whitespace-p line)
	    do (write-line line))
      (force-output))))

(defun head-request (client vnc-port http-version)
  "Open a connection to the VNC sever running at port VNC-PORT,
generate a session-id, write the session-id to CLIENT and log the
start of the session."
  (let* ((fd (connect-to-inet-socket "localhost" vnc-port))
	 (vnc-socket (sys:make-fd-stream fd :input t :output t))
	 (id (register-session-socket vnc-socket)))
    (format t "~&Starting session: ~A~%" id)
    (format client "HTTP/1.~D 200 OK~A" http-version +crlf+)
    (format client "x-vnc-session-id: ~D~A" id +crlf+)
    (format client "Cache-control: no-cache~A" +crlf+)
    (format client "Pragma: no-cache~A" +crlf+)
    (format client "Connection: close~A" +crlf+)
    (format client "Content-length: 4~A~A" +crlf+ +crlf+)
    (write-char (code-char (ldb (byte 8 0) id))  client)
    (write-char	(code-char (ldb (byte 8 8) id))  client)
    (write-char	(code-char (ldb (byte 8 16) id)) client)
    (write-char	(code-char (ldb (byte 8 24) id)) client)
    (log-session-start client id)
    (finish-output client)
    (ldebug infrequent "~%")
    ))

(defun select-method (client vnc-port)
  "Read the next request from CLIENT and decide how to respond to the
request."
  (unwind-protect 
       (handler-case 
	   (loop 
	    (multiple-value-bind (method id version keep-alive content-length)
		(parse-request-header client)
	      (ecase method
		(get-session-id
		 (head-request client vnc-port version)
		 (return))
		(get
		 (get-request client (lookup-session-socket id) version)
		 (unless keep-alive
		   (return)))
		(post
		 (post-request client content-length
			       (lookup-session-socket id) 
			       version)
		 (unless keep-alive
		   (return))))))
	 (end-of-file () 
	   (ldebug infrequent "~&  End-of-File. Closing: ~A~%" client))
	 (sys:io-timeout ()
	   (ldebug infrequent "~&  Timeout. Closing: ~A~%" client))
	 (condition (condition)
	   (format *debug-io* "~
~&  Condition: ~A
    Closing connection: ~A~%" condition client)))
    (close client)))

(defun accept-loop (listener vnc-port)
  "Accept connections on socket LISTENER and spawn a new thread for
each client."
  (unwind-protect
       (loop 
	(let ((fd (accept-tcp-connection listener)))
	  (let ((client (sys:make-fd-stream fd :input t :output t
					    :timeout 15)))
	    (ldebug accept "~&accepted connection: ~a~%" client)
	    (make-process (lambda () (select-method client vnc-port))
			  :name "dispatcher")
	    (mp:process-yield))))
    (close-socket listener)
    (loop for (id . socket) in *vnc-sockets* do (close socket))
    (setq *vnc-sockets* '())
    (ldebug infrequent "terminating accept-loop.~%")))

(defun forward-server (from-port to-port)
  "Start the forward server.  Listen on port FROM-PORT and connect to
the VNC server running on TO-PORT."
  (let ((listener (create-inet-listener from-port :stream :reuse-address t)))
    (format t "; Accepting connections on port: ~A.~%" from-port)
    (format t "; Type C-c to interrupt the forwarder.~%")
    (accept-loop listener to-port)))

#+(or)
(progn
  (profile:profile-all)
  (profile:reset-time))

;;; (profile:unprofile)
;;; (profile:report-time)

;; (forward-server 4567 5902)