/* Forwarder.java --- Server part of the Rfb-over-HTTP tunnel.

   Copyright (C) 2003  Helmut Eller <helmut@online-marketwatch.com>

   Part of HttpTunnel4vnc
   (see http://www.online-marketwatch.com/HttpTunnel4vnc/) */


import java.io.*;
import java.net.*;
import java.util.HashMap;

/**
 * @version 0.2
 **/
final class Forwarder {
    int vncPort;
    int fromPort;
    HashMap sessions = new HashMap ();
    
    Forwarder (int from, int to) { fromPort = from; vncPort = to; }

    private final static void debug (String string) {
	System.out.println (string);
    }

    int registerSessionSocket (Sock s) {
	synchronized (sessions) {
	    int id = sessions.size ();
	    sessions.put (new Integer (id), s);
	    return id;
	}
    }

    Sock lookupSessionSocket (int id) {
	synchronized (sessions) {
	    Sock server = (Sock)sessions.get (new Integer (id));
	    if (server == null)
		throw new Error ("No socket for id: " + id);
	    return server;
	}
    }

    static boolean containsOnlyWithespace (byte[] b, int length) {
	for (int i = 0; i < length; i++) 
	    if (! Character.isWhitespace ((char)b[i]))
		return false;
	return true;
    }

    static int parseInt (String s, int start) {
	int value = 0;
	int max = s.length ();
	int i = start;
	for (;;) {
	    if (i == max)
		break;
	    char c = s.charAt (i);
	    if (c == ' ' || c == '\r')
		break;
	    if (!Character.isDigit (c))
		throw new Error ("Junk read: " + c);
	    value = (value * 10) + Character.digit (c, 10);
	    i++;
	}
	if (i == start)
	    throw new Error ("ParseInt failed");
	// debug ("parseInt: " + s + "[" + start + "] => " + value);
	return value;
    }

    static int readLine (BufferedInputStream in, byte[] buffer) 
	throws IOException {
	int max = buffer.length;
	int i = 0;
	for (;;) {
	    if (i == max)
		throw new Error ("Buffer to small");
	    byte c = (byte)in.read ();
	    if (c == '\n')
		return i;
	    if (c == -1)
		throw new EOFException ();
	    buffer[i] = c;
	    i++;
	}
    }

    static final class Sock {
	Socket socket;
	BufferedInputStream in;
	BufferedOutputStream out;
	Sock (Socket s) throws IOException { 
	    socket = s;
	    in = new BufferedInputStream (s.getInputStream ());
	    out = new BufferedOutputStream (s.getOutputStream ());
	}
	public String toString () { return socket.toString (); }
    }

    final class Dispatcher extends Thread {
	Sock client;
	Dispatcher (Sock s) throws IOException { client = s; }
	String readRequestLine () throws IOException {
	    BufferedInputStream in = client.in;
	    byte[] buffer = new byte[200];
	    for (;;) {
		int size = readLine (in, buffer);
		if (!containsOnlyWithespace (buffer, size))
		    return new String (buffer, 0, size);
	    }
	}
	void startSession () throws IOException {
	    Sock s = new Sock (new Socket ("localhost", vncPort));
	    int id = registerSessionSocket (s);
	    BufferedOutputStream out = client.out;
	    out.write ("HTTP/1.1 200 OK\r\n".getBytes ());
	    out.write (("x-vnc-session-id: " + id + "\r\n").getBytes ());
	    out.write ("Cache-control: no-cache\r\n".getBytes ());
	    out.write ("Connection: close\r\n".getBytes ());
	    out.write ("Content-length: 4\r\n\r\n".getBytes ());
	    out.write ((byte)((id >>  0) & 0xff));
	    out.write ((byte)((id >>  8) & 0xff));
	    out.write ((byte)((id >> 16) & 0xff));
	    out.write ((byte)((id >> 24) & 0xff));
	    out.flush ();
	    client.socket.close ();
	    System.out.println ("; Starting session " + id + " / "+ client);
	}
	void skipToBody () throws IOException {
	    byte[] buffer = new byte[200];
	    BufferedInputStream in = client.in;
	    for (;;) {
		int size = readLine (in, buffer);
		if (containsOnlyWithespace (buffer, size))
		    break;
	    }
	}
	int parseContentLength () throws IOException {
	    for (;;) {
		String line = readRequestLine ();
		final String cl = "Content-length: ";
		final int length = cl.length ();
		if (line.regionMatches (true, 0, cl, 0, length))
		    return parseInt (line, length);
	    }
	}
	void copyToClient (Sock server) throws IOException {
	    BufferedOutputStream out = client.out;
	    BufferedInputStream in  = server.in;
	    out.write ("HTTP/1.1 200 OK\r\n".getBytes ());
	    out.write ("Cache-control: no-cache\r\n".getBytes ());
	    out.write ("Connection: keep-alive\r\n".getBytes ());
	    byte[] buffer = new byte[50000];
	    int length = buffer.length;
	    int count = in.read (buffer, 0, length);
	    if (count == -1)
		throw new Error ("Premature EOF");
	    if (count < length) {
		out.write ("Content-length: ".getBytes ());
		out.write (String.valueOf (count).getBytes ());
		out.write ("\r\n\r\n".getBytes ());
		out.write (buffer, 0, count);
		out.flush ();
	    } else {
		out.write ("Transfer-Encoding: chunked\r\n\r\n" .getBytes ());
		for (;;) {
		    out.write (Integer.toHexString (count).getBytes ());
		    out.write ("\r\n".getBytes ());
		    out.write (buffer, 0, count);
		    out.write ("\r\n".getBytes ());
		    if (count < length)
			break;
		    count = in.read (buffer, 0, length);
		}
		out.write ("0\r\n\r\n".getBytes ());
		out.flush ();
	    }
	    // System.out.write (buffer, 0, count);
	}
	void copyBytes (BufferedInputStream in, BufferedOutputStream out,
			int length) throws IOException {
	    int pending = length;
	    byte[] buffer = new byte[length];
	    while (pending > 0) {
		int count = in.read (buffer, 0, length);
		if (count == -1) 
		    throw new Error ("Premature EOF");
		out.write (buffer, 0, count);
		// System.out.write (buffer, 0, count);
		pending -= count;
	    }
	    out.flush ();
	}
	void copyToServer (Sock server, int contentLength) throws IOException {
	    BufferedOutputStream out = client.out;
	    out.write ("HTTP/1.1 200 OK\r\n".getBytes ());
	    out.write ("Content-length: 0\r\n\r\n".getBytes ());
	    out.flush ();
	    copyBytes (client.in, server.out, contentLength);
	}
	boolean dispatch () throws IOException {
	    String request = readRequestLine ();
	    if (request.startsWith ("GET /get-session-id ")) {
		startSession ();
		return false;
	    } else if (request.startsWith ("GET /vnc-session?id=")) {
		int id = parseInt (request, "GET /vnc-session?id=".length ());
		skipToBody ();
		copyToClient (lookupSessionSocket (id));
		return true;
	    } else if (request.startsWith ("POST /vnc-session?id=")) {
		int id = parseInt (request, "POST /vnc-session?id=".length ());
		int length = parseContentLength ();
		skipToBody ();
		copyToServer (lookupSessionSocket (id), length);
		return true;
	    } else 
		throw new Error ("Unsupported HTTP-method: " + request);
	}
	public void run () { 
	    // debug ("thread for " + client);
	    try {
		while (dispatch ()); 
	    } catch (EOFException e) {
	    } catch (Exception e) {
		System.err.println ("Error in dispatch thread: " + e);
		e.printStackTrace ();
	    } finally {
		try { client.socket.close (); } catch (Exception e) {};
	    }
	}
    }

    void listen () throws IOException {
	ServerSocket s = new ServerSocket (fromPort);
	System.out.println ("; Accepting connections on port: " + fromPort);
	for (;;) 
	    new Dispatcher (new Sock (s.accept ())).start ();
    }

    static void usage () {
	System.out.println ("java Forwarder <from-port> <to-port>");
	System.exit (1);
    }

    public static void main (String args[]) throws Exception {
	if (args.length != 2) 
	    usage ();
	int from = Integer.parseInt (args[0]);
	int to = Integer.parseInt (args[1]);
	new Forwarder (from, to).listen ();
    }
} 
